<?php
// Permitir peticiones desde cualquier origen (CORS)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=UTF-8');

// Responder a peticiones OPTIONS (pre-flight)
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// 1. --- OBTENER CONFIGURACIÓN (TOKEN) DESDE LA BD ---
require_once 'conexion.php';

$sql_config = "SELECT plan FROM tm_configuracion LIMIT 1";
$result_config = mysqli_query($conn, $sql_config);

if (!$result_config || mysqli_num_rows($result_config) == 0) {
    http_response_code(500);
    echo json_encode(['error' => 'No se pudo obtener la configuración del sistema.']);
    mysqli_close($conn);
    exit;
}

$config_row = mysqli_fetch_assoc($result_config);
$plan_json = json_decode($config_row['plan'], true);

if (json_last_error() !== JSON_ERROR_NONE || !isset($plan_json['wsp_token'])) {
    http_response_code(500);
    echo json_encode(['error' => 'La configuración de WhatsApp es inválida o no contiene un token.']);
    mysqli_close($conn);
    exit;
}

$apiToken = $plan_json['wsp_token'];

// 2. --- RECIBIR Y DECODIFICAR DATOS DEL PEDIDO ---
$jsonPayload = file_get_contents('php://input');
$pedido = json_decode($jsonPayload, true);

if (!$pedido || !isset($pedido['cliente']) || !isset($pedido['productos']) || !isset($pedido['cliente']['telefono'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos del pedido inválidos o falta el teléfono del cliente.']);
    mysqli_close($conn);
    exit;
}

// 2.1 --- GUARDAR EL PEDIDO EN LA BASE DE DATOS (PERSISTENCIA) ---
$cliente = $pedido['cliente'];
$productos = $pedido['productos'];
$total_num = floatval(str_replace(['S/', ' ', ','], '', $pedido['total']));

$stmt = mysqli_prepare($conn, "INSERT INTO tm_pedido_web (cliente_nombre, cliente_telefono, cliente_direccion, cliente_referencia, cliente_comentario, productos_json, total) VALUES (?, ?, ?, ?, ?, ?, ?)");

$productos_json = json_encode($productos);
$referencia = $cliente['referencia'] ?? null;
$comentario = $cliente['comentario'] ?? null;

mysqli_stmt_bind_param($stmt, "ssssssd", 
    $cliente['nombre'], 
    $cliente['telefono'], 
    $cliente['direccion'], 
    $referencia, 
    $comentario, 
    $productos_json, 
    $total_num
);


if (!mysqli_stmt_execute($stmt)) {
    // Si falla la base de datos, registramos el error pero intentamos seguir con el WhatsApp
    error_log("Error al persistir pedido en DB: " . mysqli_stmt_error($stmt));
}
mysqli_stmt_close($stmt);

// 3. --- FORMATEAR EL MENSAJE PARA WHATSAPP ---
$cliente = $pedido['cliente'];
$productos = $pedido['productos'];
$total = $pedido['total'];

// ***** MENSAJE DE CONFIRMACIÓN PARA EL CLIENTE (ACTUALIZADO) *****
$mensaje = "✅ *¡Hola " . $cliente['nombre'] . "! Hemos recibido tu pedido.* ✅\n\n";
$mensaje .= "Estamos preparando tus productos y pronto estarán en camino a la siguiente dirección:\n";
$mensaje .= "📍 *Direccion:*  " . $cliente['direccion'] . "\n";

// ***** CAMBIO CLAVE AQUÍ: Añadir Referencia y Comentario si existen *****
if (!empty($cliente['referencia'])) {
    $mensaje .= "▪️ *Referencia:* " . $cliente['referencia'] . "\n";
}
if (!empty($cliente['comentario'])) {
    $mensaje .= "▪️ *Tu comentario:* " . $cliente['comentario'] . "\n";
}
$mensaje .= "\n";


$mensaje .= "📦 *RESUMEN DE TU PEDIDO:*\n";
foreach ($productos as $producto) {
    $subtotal = number_format($producto['precio'] * $producto['cantidad'], 2, '.', '');
    $mensaje .= "▪️ (" . $producto['cantidad'] . "x) " . $producto['presentacion'] . " - S/ " . $subtotal . "\n";
}

$mensaje .= "\n💰 *TOTAL A PAGAR:* " . $total . "\n\n";
$mensaje .= "¡Gracias por tu preferencia!";

// 4. --- PREPARAR Y ENVIAR LA PETICIÓN A LA API DE WHATSAPP ---
$urlApi = 'https://efasis.efa-api.com/api/message/send-text';
$numeroCliente = $cliente['telefono'];

$numeroLimpio = preg_replace('/\D/', '', $numeroCliente);
if (strlen($numeroLimpio) == 9) {
    $numeroLimpio = '51' . $numeroLimpio;
}

$payload = json_encode([
    'number' => $numeroLimpio,
    'message' => $mensaje
]);

$ch = curl_init($urlApi);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $apiToken,
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

// 5. --- ENVIAR RESPUESTA AL FRONTEND ---
if ($httpCode >= 200 && $httpCode < 300) {
    http_response_code(200);
    echo json_encode(['success' => true, 'message' => 'Confirmación enviada al cliente por WhatsApp.']);
} else {
    $apiError = json_decode($response, true);
    $errorMessage = isset($apiError['error']['response']['message'][0]['exists']) && $apiError['error']['response']['message'][0]['exists'] === false
        ? 'El número de teléfono proporcionado no tiene WhatsApp.'
        : 'Error al enviar la confirmación por WhatsApp.';

    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $errorMessage, 'http_code' => $httpCode, 'api_response' => $response]);
}

mysqli_close($conn);
?>