﻿document.addEventListener("DOMContentLoaded", () => {
    // --- URLs Y CONSTANTES ---
    const API_EMPRESA = 'Database/DataEmpresa.php';
    const API_PRODUCTOS = 'Database/DataBase.php';
    const LOGO_URL_BASE = '../../facturacion/public/images/';
    const Img_URL = '../../facturacion/public/images/productos/';

    // --- SELECTORES DEL DOM ---
    const logoNav = document.getElementById("logo-nav"),
        nombreNav = document.getElementById("nombre-nav"),
        logoFooter = document.getElementById("logo-footer"),
        nombreFooter = document.getElementById("nombre-footer"),
        contactosFooter = document.getElementById("contactos-footer"),
        mapContainer = document.getElementById("map-container"),
        contenedorPopulares = document.querySelector("#contenedor-populares"),
        heroTitle = document.getElementById("hero-title"),
        header = document.querySelector(".header-inicio");

    // --- EFECTO SCROLL HEADER ---
    window.addEventListener("scroll", () => {
        if (header) {
            header.classList.toggle("header-scrolled", window.scrollY > 50);
        }
    });

    // --- FUNCIONES DE CARGA DE DATOS ---
    async function cargarDatosEmpresa() {
        try {
            const response = await fetch(API_EMPRESA);
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            const data = await response.json();

            if (data && data.error) {
                throw new Error(data.error);
            }

            if (data && data.length > 0) {
                const empresa = data[0];
                const nombreComercial = empresa.nombre_comercial || "Nombre no disponible";

                document.title = `${nombreComercial} - Sabor que Une`;

                // --- Logo y Nombre ---
                if (empresa.logo) {
                    const logoUrl = LOGO_URL_BASE + empresa.logo;
                    if (logoNav) { logoNav.src = logoUrl; logoNav.style.display = 'inline-block'; if (nombreNav) nombreNav.style.display = 'none'; }
                    if (logoFooter) { logoFooter.src = logoUrl; logoFooter.style.display = 'block'; if (nombreFooter) nombreFooter.style.display = 'none'; }
                } else {
                    if (nombreNav) nombreNav.innerText = nombreComercial;
                    if (nombreFooter) nombreFooter.innerText = nombreComercial;
                }

                // --- Hero Title ---
                if (heroTitle) { heroTitle.innerHTML = `El Aut&eacute;ntico Sabor de <br>${nombreComercial}`; }

                // --- Contactos Footer ---
                if (contactosFooter) {
                    let htmlContactos = "";
                    if (empresa.direccion_comercial) {
                        htmlContactos += `<li><i class="bi bi-geo-alt-fill me-2"></i> ${empresa.direccion_comercial}</li>`;
                    }

                    const telf = empresa.celular || empresa.telefono;
                    if (telf) {
                        htmlContactos += `<li><i class="bi bi-telephone-fill me-2"></i> ${telf}</li>`;
                    }

                    const mail = empresa.email || empresa.correo;
                    if (mail) {
                        htmlContactos += `<li><i class="bi bi-envelope-fill me-2"></i> ${mail}</li>`;
                    }
                    contactosFooter.innerHTML = htmlContactos || '<li>Informaci&oacute;n no disponible</li>';
                }

                // --- Mapa (Automatizado y Preciso) ---
                if (empresa.direccion_comercial && mapContainer) {
                    // Concatenamos para evitar errores de ubicación (ej: Tingo María vs Lima)
                    const direccionCompleta = [
                        empresa.direccion_comercial,
                        empresa.distrito,
                        empresa.provincia,
                        empresa.departamento,
                        "Peru"
                    ].filter(Boolean).join(", ");

                    const direccionCodificada = encodeURIComponent(direccionCompleta);
                    mapContainer.innerHTML = `<iframe src="https://maps.google.com/maps?q=${direccionCodificada}&t=&z=16&ieUTF8&iwloc=&output=embed" frameborder="0" scrolling="no" marginheight="0" marginwidth="0"></iframe>`;
                } else if (mapContainer) {
                    mapContainer.innerHTML = `<p class="text-center text-muted">Ubicaci&oacute;n no disponible.</p>`;
                }

                // --- WhatsApp Reserva ---
                const btnReserva = document.getElementById("btn-reserva-whatsapp");
                const numWsp = empresa.celular || empresa.telefono;
                if (btnReserva && numWsp) {
                    const numeroLimpio = numWsp.replace(/\D/g, '');
                    const numeroWhatsapp = `51${numeroLimpio}`;
                    btnReserva.href = `https://wa.me/${numeroWhatsapp}?text=Hola,%20quisiera%20hacer%20una%20reserva.`;
                }
            } else {
                if (nombreNav) nombreNav.innerText = "Restaurante";
                if (nombreFooter) nombreFooter.innerText = "Restaurante";
                if (contactosFooter) contactosFooter.innerHTML = "<li>Informaci&oacute;n no disponible</li>";
            }
        } catch (error) {
            console.error("Error al cargar datos de la empresa:", error);
            if (nombreNav) nombreNav.innerText = "Error";
            if (nombreFooter) nombreFooter.innerText = "Error";
            if (contactosFooter) contactosFooter.innerHTML = `<li>Error en el servidor.</li>`;
        }
    }

    async function cargarPopulares() {
        if (!contenedorPopulares) return;

        // --- Mostrar Skeletons ---
        contenedorPopulares.innerHTML = Array(3).fill(0).map(() => `
            <div class="swiper-slide">
                <div class="card popular-card h-100">
                    <div class="skeleton skeleton-img"></div>
                    <div class="popular-card-body">
                        <div class="skeleton skeleton-title"></div>
                        <div class="skeleton skeleton-text"></div>
                    </div>
                </div>
            </div>`).join('');

        try {
            const response = await fetch(API_PRODUCTOS);
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            let productos = await response.json();

            if (productos && productos.length > 0) {
                contenedorPopulares.innerHTML = "";
                const populares = productos.slice(0, 6);
                populares.forEach(producto => {
                    const imgUrl = producto.archivo_imagen ? (Img_URL + producto.archivo_imagen) : './img/fallback-producto.jpg';
                    const slide = document.createElement("div");
                    slide.className = "swiper-slide";
                    slide.innerHTML = `
                        <div class="card popular-card h-100">
                            <img src="${imgUrl}" class="popular-card-img" alt="${producto.presentacion}" onerror="this.src='./img/fallback-producto.jpg'">
                            <div class="popular-card-body">
                                <h5 class="popular-card-title">${producto.presentacion}</h5>
                                <div class="d-flex justify-content-between align-items-center mt-2">
                                    <p class="popular-card-price mb-0">S/ ${parseFloat(producto.precio).toFixed(2)}</p>
                                    <a href="menu.html" class="popular-card-btn"><i class="bi bi-arrow-right"></i></a>
                                </div>
                            </div>
                        </div>`;
                    contenedorPopulares.appendChild(slide);
                });

                new Swiper('.populares-swiper', {
                    slidesPerView: 1,
                    spaceBetween: 30,
                    navigation: { nextEl: '.swiper-button-next', prevEl: '.swiper-button-prev' },
                    breakpoints: { 768: { slidesPerView: 2 }, 1200: { slidesPerView: 3 } }
                });
            } else {
                contenedorPopulares.innerHTML = `<p class="text-center text-muted w-100">No hay platos populares disponibles por ahora.</p>`;
            }
        } catch (error) {
            console.error("Error al cargar populares:", error);
            contenedorPopulares.innerHTML = `<div class="alert alert-danger w-100">No se pudieron cargar los platos populares.</div>`;
        }
    }

    // --- EJECUCION ---
    cargarDatosEmpresa();
    cargarPopulares();
});